/**  @file bta_rms_frames.c
*
*    @brief This file implements the filter (see header)
*
*    BLT_DISCLAIMER
*
*    @author Michael Koschutnig
*
*    @cond svn
*
*    Information of last commit
*    $Rev::               $:  Revision of last commit
*    $Author::            $:  Author of last commit
*    $Date::              $:  Date of last commit
*
*    @endcond
*/


#include <bta.h>
#include "bta_rms_frames.h"
#include <stdlib.h>
#include <math.h>

#ifndef BTA_EXCLUDE_FILTERS

BTA_Status BFLTrmsFramesInit(BTA_FltRmsFramesConfig *config, BTA_FltHandle *handle, BTA_InfoEventInst *infoEventInst) {
    if (!handle || !config) {
        return BTA_StatusInvalidParameter;
    }
    *handle = 0u;
    if (config->windowLength < 2u) {
        return BTA_StatusInvalidParameter;
    }
    BTA_FltRmsFramesInst *inst = (BTA_FltRmsFramesInst *)calloc(1, sizeof(BTA_FltRmsFramesInst));
    if (!inst) {
        return BTA_StatusOutOfMemory;
    }
    inst->windowLength = config->windowLength;
    inst->channelToProcess = config->channelToProcess;
    inst->channelIdResult = config->channelIdResult;
	inst->frameRef = 0u;

    inst->infoEventInst = infoEventInst;
    *handle = inst;
    BTAinfoEventHelper(inst->infoEventInst, 5, BTA_StatusInformation, "Filter rms frames: init finished", 0);
    return BTA_StatusOk;
}


BTA_Status BFLTrmsFramesClose(BTA_FltHandle *handle) {
    BTA_FltRmsFramesInst **inst = (BTA_FltRmsFramesInst **)handle;
    uint32_t i = 0u;
    free((*inst)->frameRef);
    (*inst)->frameRef = 0u;
    if ((*inst)->frameSum) {
        for (i = 0u; i < (*inst)->frameSum->channelsLen; i++) {
            free((*inst)->frameSum->channels[i]->data);
            (*inst)->frameSum->channels[i]->data = 0u;
        }
        free((*inst)->frameSum);
        (*inst)->frameSum = 0u;
    }
    free(*inst);
    *inst = 0u;
    return BTA_StatusOk;
}


BTA_Status BFLTrmsFramesApply(BTA_FltHandle handle, BTA_Frame **frame) {
    BTA_FltRmsFramesInst *inst = (BTA_FltRmsFramesInst *)handle;
    uint32_t i = 0u;
    uint32_t j = 0u;

    if (!inst || !frame) {
        return BTA_StatusInvalidParameter;
    }
    if (!*frame) {
        return BTA_StatusInvalidParameter;
    }

    if (0u == inst->frameRef) {
        BTAcloneFrame(*frame, &inst->frameRef);
    }
    else {

        if ((*frame)->channelsLen != inst->frameRef->channelsLen) {
            BTAfreeFrame(&inst->frameRef);
            BTAcloneFrame(*frame, &inst->frameRef);
            if (inst->frameSum) {
                for (i = 0u; i < inst->frameSum->channelsLen; i++) {
                    free(inst->frameSum->channels[i]->data);
                    inst->frameSum->channels[i]->data = 0u;
                }
                free(inst->frameSum);
                inst->frameSum = 0u;
            }
            BTAinfoEventHelper(inst->infoEventInst, 5, BTA_StatusInformation, "Filter rms frames: channelsLen changed", (*frame)->channelsLen);
        }
        else {
            for (i = 0u; i < (*frame)->channelsLen; i++) {
                // frames must be totally consistent because of memory accesses
                if ((*frame)->channels[i]->id != inst->frameRef->channels[i]->id &&
                    (*frame)->channels[i]->dataFormat != inst->frameRef->channels[i]->dataFormat &&
                    (*frame)->channels[i]->dataLen != inst->frameRef->channels[i]->dataLen &&
                    (*frame)->channels[i]->unit != inst->frameRef->channels[i]->unit &&
                    (*frame)->channels[i]->xRes != inst->frameRef->channels[i]->xRes &&
                    (*frame)->channels[i]->yRes != inst->frameRef->channels[i]->yRes) {
                    BTAfreeFrame(&inst->frameRef);
                    BTAcloneFrame(*frame, &inst->frameRef);
                    if (inst->frameSum) {
                        for (j = 0u; inst->frameSum->channelsLen; j++) {
                            free(inst->frameSum->channels[j]->data);
                            inst->frameSum->channels[j]->data = 0u;
                        }
                        free(inst->frameSum);
                        inst->frameSum = 0u;
                    }
                    inst->frameCounter = 0u;
                    BTAinfoEventHelper(inst->infoEventInst, 5, BTA_StatusInformation, "Filter rms frames: frame differs from first frame in filter window", i);
                }
            }
        }
    }
    if (!inst->frameSum) {
        inst->frameSum = (BTA_Frame *)calloc(1u, sizeof(BTA_Frame));
        for (i = 0u; i < (*frame)->channelsLen; i++) {
            float *data = 0u;
            if (!inst->channelToProcess || (*frame)->channels[i]->id == inst->channelToProcess) {
                data = (float *)calloc((*frame)->channels[i]->xRes * (*frame)->channels[i]->yRes, sizeof(float));
            }
            BTAinsertChannelDataIntoFrame(inst->frameSum, (*frame)->channels[i]->id, (*frame)->channels[i]->xRes, (*frame)->channels[i]->yRes, BTA_DataFormatFloat32, (*frame)->channels[i]->unit, (*frame)->channels[i]->integrationTime, (*frame)->channels[i]->modulationFrequency, (uint8_t *)data, (*frame)->channels[i]->xRes * (*frame)->channels[i]->yRes * sizeof(float));
        }
        inst->frameCounter = 0u;
    }
    for (i = 0u; i < (*frame)->channelsLen; i++) {
        if (!inst->channelToProcess || (*frame)->channels[i]->id == inst->channelToProcess) {
            for (j = 0u; j < (uint32_t)((*frame)->channels[i]->xRes * (*frame)->channels[i]->yRes); j++) {
                ((float *)inst->frameSum->channels[i]->data)[j] += (((int16_t *)(*frame)->channels[i]->data)[j] * ((int16_t *)(*frame)->channels[i]->data)[j]);
                if (inst->frameCounter == (inst->windowLength - 1u)) {
                    //Window Length reached, compose output
                    ((float *)inst->frameSum->channels[i]->data)[j] = sqrtf(((float *)inst->frameSum->channels[i]->data)[j] / (float)inst->windowLength);
                }
            }
            if (inst->frameCounter == (inst->windowLength - 1u)) {
                if (inst->channelIdResult) {
                    inst->frameSum->channels[i]->id = inst->channelIdResult;
                    BTAinsertChannelIntoFrame(*frame, inst->frameSum->channels[i]);
                    inst->frameSum->channels[i] = 0u;
                }
                else {
                    free((*frame)->channels[i]->data);
                    free((*frame)->channels[i]);
                    (*frame)->channels[i] = inst->frameSum->channels[i];
                    inst->frameSum->channels[i] = 0u;
                }
            }
        }
    }
    
    inst->frameCounter++;
    if (inst->frameCounter == inst->windowLength) {
        free(inst->frameSum);
        inst->frameSum = 0u;
        free(inst->frameRef);
        inst->frameRef = 0u;
        inst->frameCounter = 0u;
    }
    return BTA_StatusOk;
}

#endif
